<?php

namespace App\Filament\Resources;

use App\Enums\Order\OrderStatus;
use App\Enums\Product\ProductType;
use App\Filament\Resources\OrderResource\Pages;
use App\Filament\Resources\OrderResource\RelationManagers;
use App\Models\Book;
use App\Models\Course;
use App\Models\Event;
use App\Models\Order;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\Indicator;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class OrderResource extends Resource
{
    protected static ?string $model = Order::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('user_id')
                    ->getStateUsing(function ($record) {

                        return $record->user->first_name;
                    })
                    ->label('User Name')
                    ->searchable(query: function ($query, $search) {
                        return $query->whereHas('user', function ($query) use ($search) {
                            $query->where('first_name', 'like', "%{$search}%");
                        });
                    }),

                Tables\Columns\TextColumn::make('Email')
                    ->getStateUsing(function ($record) {
                        return $record->user->email;
                    })
                    ->label('Email')
                    ->searchable(query: function ($query, $search) {
                        return $query->whereHas('user', function ($query) use ($search) {
                            $query->where('email', 'like', "%{$search}%");
                        });
                    }),

                Tables\Columns\TextColumn::make('status')
                    ->formatStateUsing(function (TextColumn $column): ?string {
                        return OrderStatus::getStatus($column->getRecord()->status);
                    })
                    ->label('status')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('amount')
                    ->numeric()
                    ->sortable(),

                Tables\Columns\TextColumn::make('productable_id')
                    ->formatStateUsing(function (TextColumn $column): ?string {

                        return $column->getRecord()->productable?->name_en;
                    })
                    ->label('Product Name')
                    ->searchable(),

                Tables\Columns\TextColumn::make('productable_type')
                    ->formatStateUsing(function (TextColumn $column): ?string {
                        return ProductType::getProductType($column->getRecord()->productable_type);
                    })
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Filter::make('User')
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->whereHas('user', function (Builder $query) use ($data) {
                            if (isset($data['first_name'])) {
                                $query->where('first_name', 'like', "%{$data['first_name']}%");
                            }
                            if (isset($data['email'])) {
                                $query->where('email', 'like', "%{$data['email']}%");
                            }
                        });
                    })
                    ->form([
                        Forms\Components\TextInput::make('first_name')
                            ->label('User Name'),
                        Forms\Components\TextInput::make('email')
                            ->label('Email'),
                    ])->indicateUsing(function (array $data): ?array {

                        $indicators = [];
                        if ($data['first_name'] ?? null) {
                            $indicators[] = Indicator::make('User name ' . $data['first_name']);

                        }

                        if ($data['email'] ?? null) {
                            $indicators[] = Indicator::make('Email ' . $data['email']);

                        }
                        return $indicators;
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('suspend')
                    ->label(function (Order $record) {
                        return $record->is_suspended ? 'Reactivate Order' : 'Suspend Order';
                    })
                    ->color('primary')
                    ->requiresConfirmation()
                    ->icon(function (Order $record) {
                        return $record->is_suspended ? 'heroicon-o-check-circle' : 'heroicon-o-x-circle';
                    })
                    ->action(function (Order $record) {
                        $record->is_suspended = !$record->is_suspended;
                        $record->save();
                    }),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('user_id')
                    ->relationship('user', 'first_name')
                    ->searchable()
                    ->required(),
                Forms\Components\TextInput::make('status')
                    ->default(OrderStatus::PAID->value)
                    ->hidden()
                    ->required(),
                Forms\Components\Select::make('productable_id')
                    ->label('Product Name')
                    ->options(function () {
                        $courses = Course::all()->pluck('name_en', 'id');
                        $books = Book::all()->pluck('name_en', 'id');
                        $events = Event::all()->pluck('name_en', 'id');
                        return $courses->merge($books)->merge($events);
                    })
                    ->required()
                    ->reactive()
                    ->afterStateUpdated(function (callable $set, $state) {
                        $productableModel = null;
                        if ($state) {
                            $course = Course::find($state);
                            $book = Book::find($state);
                            $event = Event::find($state);

                            if ($course) {
                                $productableModel = Course::class;
                            } elseif ($book) {
                                $productableModel = Book::class;
                            } elseif ($event) {
                                $productableModel = Event::class;
                            }
                        }

                        $set('productable_type', $productableModel);
                    }),
                Forms\Components\Hidden::make('productable_type')
                    ->required(),
                Forms\Components\TextInput::make('amount')
                    ->numeric(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListOrders::route('/'),
            'create' => Pages\CreateOrder::route('/create'),
            'edit' => Pages\EditOrder::route('/{record}/edit'),
        ];
    }


    public static function canDeleteAny(): bool
    {
        return false; // TODO: Change the autogenerated stub
    }
}
